package easik.sketch.util.Export.Components;

import java.util.ArrayList;

import easik.sketch.util.Export.ExportConstants;

/**
 * Class to store edge information.  Should only be used for exportation to a database.  
 * Does not provide functionality for graphical display.
 * 
 * @author Vera Ranieri 2006
 * @since 2006-05-18 Vera Ranieri
 * @version 2006-05-22 Vera Ranieri
 */
public class ExportEdge {
		/**
		 * The name of the edge
		 */
		private String _name;
		/**
		 * The source of the edge
		 */
		private String _source;
		/**
		 * The target of the edges
		 */
		private String _target;
		/**
		 * Stores whether the edge is injective
		 */
		private boolean _injective;
		
		/**
		 * Constructor.  The target of the edge is not allowed to have whitespaces.
		 * All contained whitespace is replaced by an underscore.
		 * @param n The name of the edge
		 * @param injective true if edge is injective, false otherwise
		 * @param source The name of the source table
		 * @param target The name of the target table
		 */
		public ExportEdge(String n, boolean injective, String source, String target){
			_name = n;
			_injective = injective;
			_source = source.trim().replace(' ', '_');
			_target = target.trim().replace(' ', '_');
		}
		
		/**
		 * Getter method for the name of the edge
		 * @return The name of the edge
		 */
		public String getName(){
			return _name;
		}
		
		/**
		 * Determines whether an edge is injective.
		 * 
		 * @return true if edge is injective, false otherwise.
		 */
		public boolean isInjective(){
			return _injective;
		}

		/**
		 * Getter method of the name of the target (that is, codomain) of this edge.
		 * @return The target table of the edge
		 */
		public String getTarget() {
			return _target;
		}
		
	
		/**
		 * Getter method of the name of the source (that is, domain) of this edge.
		 * @return The source table of the edge
		 * @since 2006-05-22
		 */
		public String getSource(){
			return _source;
		}
		
		/**
		 * Formats the foreign key string for SQL.  Accepts a table as a parameter.  The edges of this table are then 
		 * used to determine the foreign key string.  The foreign key string does not contain any attribute or primary/
		 * unique key definitions, exclusive of unique keys defined because of an injective edge.
		 * 
		 * @param t The table for which the foreign key string should be determined
		 * @return The foreign key string formatted for SQL derived from table t
		 * @since 2006-06-14
		 */
		public static String getForeignKeyString(ExportTable t){
			ArrayList foreignKeys = t.getEdges();
			int size = foreignKeys.size();
			
			String foreignKeyString = "";
			
			ExportEdge e;
			
			for(int i = 0; i< size; i++){
				e = (ExportEdge)foreignKeys.get(i);
				foreignKeyString += ExportConstants.FOREIGN + e.getTarget() +ExportConstants.ID+ ") " 
									+ ExportConstants.REF +e.getTarget() + " (" + e.getTarget() + ExportConstants.ID +") "
									+ ExportConstants.ON_UPDATE + ExportConstants.ON_DELETE + ", \n";
				if(e.isInjective()){
					foreignKeyString += ExportConstants.UNIQUE + e.getTarget() + ExportConstants.ID +"), \n";
				}
			}
			
			return foreignKeyString;
		}
}
